/*
 * Scripts generating a number for the Gauss probability density function
 * Used to interact with pgbench and setshell command
 * 
 * Dec-02-2009 By Michael Paquier, NTT-OSSC
 */


#include <math.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <stdlib.h>

#define STATUS_OK 0
#define STATUS_ERROR -1

static char *progname = "gauss";

int main(int argc, char *argv[])
{
	double pi = 4*atan(1);
	double center = 0, variance = 1, x = 0.5;
	double res = 0;
	int opt;

	if (argc > 1)
	{
		if (strcmp(argv[1], "--help") == 0 || strcmp(argv[1], "-?") == 0)
		{
			fprintf(stdout, "Be sure to put the correct options\n");
			fprintf(stdout, "Normal options for Gauss distribution\n");
			fprintf(stdout, "-x parameter based on a random one\n");
			fprintf(stdout, "-c center\n");
			fprintf(stdout, "-s variance\n");
			fprintf(stdout, "Density function g(x,c,s) is defined by:\n");
			fprintf(stdout, "g(x,c,s) = 1/(2*c*pi)^0.5*f(x)\n");
			fprintf(stdout, "with f(x,c,s) = exp(-(x-c)^2/(2*s^2))\n");
			return STATUS_ERROR;
		}
	}
	while ((opt = getopt(argc, argv, "x:c:s:")) != -1)
	{
		switch (opt)
		{	
			case 'x':
				/* float parameter normally generated randomly */
				x = atof(optarg);
				break;
			case 'c':
				/* center of Gauss law */
				center = atof(optarg);
				break;
			case 's':
				/* variance of Gauss law */
				variance = atof(optarg);
				break;

			default:
				fprintf(stdout, "Try \"%s --help\" for more information.\n",
						progname);
			}
		}
	/* In case there are undesirable options */
	if (optind < argc)
	{
		fprintf(stdout,"%s: invalid argument: \"%s\"\n",
				progname, argv[optind]);
		fprintf(stdout,"Try \"%s --help\" for more information.\n",
				progname);
		return STATUS_ERROR;
	}

	/* Check that parameters are set correctly */
	if (variance <= 0)
	{
		fprintf(stderr, "%s: error: variance cannot be null or negative...\n",
				progname);
		return STATUS_ERROR;
	}
	
	/* Main purpose of this script, just calculate the value and print the result abruptly */
	res = (x-center)/variance;
	res = -1.0*pow(res,2);
	res = exp(res);
	res = res/pow(2.0*pi*variance,0.5);
	fprintf(stdout, "%f\n", res);

	return STATUS_OK;
}
