/*
 * Scripts generating a number for the Pareto probability density function
 * Used to interact with pgbench and setshell command
 * 
 * Dec-02-2009 By Michael Paquier, NTT-OSSC
 */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <stdlib.h>

#define STATUS_OK 0
#define STATUS_ERROR -1

static char *progname = "pareto";

int main(int argc, char *argv[])
{
	double x = 1, y = 1, alpha = 1;
	double res = 0;
	int opt;

	if (argc > 1)
	{
		if (strcmp(argv[1], "--help") == 0 || strcmp(argv[1], "-?") == 0)
		{
			fprintf(stdout, "Be sure to put the correct options\n");
			fprintf(stdout, "Normal options for Pareto distribution\n");
			fprintf(stdout, "-x parameter based on a random one\n");
			fprintf(stdout, "-y pareto second parameter\n");
			fprintf(stdout, "-a Pareto index\n");
			fprintf(stdout, "Density function Pf(x,y,a) is defined by:\n");
			fprintf(stdout, "if x<y Pf(x,y,a) = 0\n");
			fprintf(stdout, "if y<x Pf(x,y,a) = a*y^a/x^(a+1)\n");
			return STATUS_ERROR;
		}
	}

	while ((opt = getopt(argc, argv, "x:a:y:")) != -1)
	{
		switch (opt)
		{	
			case 'x':
				/* float parameter normally generated randomly */
				x = atof(optarg);
				break;

			case 'a':
				/* Pareto index */
				alpha = atof(optarg);
				break;
			case 'y':
				/* if x<y, then 0 is returned */
				y = atof(optarg);
				break;

			default:
				fprintf(stdout,"Try \"%s --help\" for more information.\n",
						progname);
		}
	}

	/* In case there are undesirable options */
	if (optind < argc)
	{
		fprintf(stdout,"%s: invalid argument: \"%s\"\n",
				progname, argv[optind]);
		fprintf(stdout,"Try \"%s --help\" for more information.\n",
				progname);
		return STATUS_ERROR;
	}

	/* Check that parameters are set correctly */
	if (alpha < 1)
	{
		fprintf(stdout, "%s: error: alpha needs a value strictly superior to 1...\n",
				progname);
		return STATUS_ERROR;
	}
	if (y < 0)
	{
		fprintf(stdout, "%s: error, y has to be strictly positive...\n",
				progname);
		return STATUS_ERROR;
	}
	
	/* Main purpose of this script, just calculate the and print the result abruptly */

	if (x < y)
	{
		/* a nul value is returned */
		fprintf(stdout, "0\n");
	}
	else
	{
		res = alpha*pow(y,alpha);
		res = res/pow(x,(alpha+1.0));
		fprintf(stdout, "%f\n", res);
	}

	return STATUS_OK;
}
