/*
 * Another PoC of MemoryContext for DSA based on Thomas Munro's work
 *
 * https://www.postgresql.org/message-id/
 * CAEepm%3D22H0TDCAHWh%3DYWmSV%2BX%2BbTtcTNg8RpP%3DeaCjWJU_d-9A
 * %40mail.gmail.com
 *
 */

#include "postgres.h"

#include "fmgr.h"
#include "lib/ilist.h"
#include "miscadmin.h"
#include "nodes/pg_list.h"
#include "nodes/memnodes.h"
#include "storage/ipc.h"
#include "storage/lwlock.h"
#include "storage/shmem.h"
#include "utils/dsa.h"
#include "utils/memutils.h"

#define MY_AREA_SIZE (1024 * 1024)
#define NUM_CHUNKS 64 /* an arbitrary number */


PG_MODULE_MAGIC;

void _PG_init(void);
PG_FUNCTION_INFO_V1(hoge);
PG_FUNCTION_INFO_V1(hoge_list);
PG_FUNCTION_INFO_V1(hoge_list_error);

static void hoge_shmem_startup_hook(void);
static MemoryContext make_hoge_memory_context(dsa_area *area, void *base, bool isLocal);

static shmem_startup_hook_type prev_shmem_startup_hook;
static void *my_raw_memory;
static dsa_area *my_area;
static MemoryContext my_shared_dsa_context;
static MemoryContext my_local_dsa_context;

static List **my_list;
void SetDsaSharedContext(MemoryContext local, MemoryContext shared);
MemoryContext CreateDsaLocalContext(MemoryContext shared);

void
_PG_init(void)
{
	/* This only works if preloaded by the postmaster. */
	if (!process_shared_preload_libraries_in_progress)
		return;

	/* Request a chunk of traditional shared memory. */
	RequestAddinShmemSpace(MY_AREA_SIZE);

	/* Register our hook for phase II of initialization. */
	prev_shmem_startup_hook = shmem_startup_hook;
	shmem_startup_hook = hoge_shmem_startup_hook;
}

static void
hoge_shmem_startup_hook(void)
{
	MemoryContext	old_context;
	bool		found;

	if (prev_shmem_startup_hook)
		prev_shmem_startup_hook();

	old_context = MemoryContextSwitchTo(TopMemoryContext);

	LWLockAcquire(AddinShmemInitLock, LW_EXCLUSIVE);

	/* Allocate, or look up, a chunk of raw fixed-address shared memory. */
	my_raw_memory = ShmemInitStruct("hoge", MY_AREA_SIZE, &found);
	if (!found)
	{
		/*
		 * Create a new DSA area, and clamp its size so it can't make any
		 * segments outside the provided space.
		 */
		my_area = dsa_create_in_place(my_raw_memory, MY_AREA_SIZE, 0, NULL);
		dsa_set_size_limit(my_area, MY_AREA_SIZE);
	}
	else
	{
		/* Attach to an existing area. */
		my_area = dsa_attach_in_place(my_raw_memory, NULL);
	}

	/* Also allocate or look up a list header. */
	my_list = ShmemInitStruct("hoge_list", MY_AREA_SIZE, &found);
	if (!found)
		*my_list = NIL;

	LWLockRelease(AddinShmemInitLock);

	/* Create a memory context. */
	my_shared_dsa_context = make_hoge_memory_context(my_area, my_raw_memory, false);

	MemoryContextSwitchTo(old_context);
}

Datum
hoge(PG_FUNCTION_ARGS)
{
	char *s;
	MemoryContext old_context;

	old_context = MemoryContextSwitchTo(TopTransactionContext);
	my_local_dsa_context = CreateDsaLocalContext(my_shared_dsa_context);
	MemoryContextSwitchTo(old_context);

	old_context = MemoryContextSwitchTo(my_local_dsa_context);

	/* Simple smoke test: allocate and free immediately. */
	s = pstrdup("hello world");
	pfree(s);
	SetDsaSharedContext(my_local_dsa_context, my_shared_dsa_context);
	/* do it again */
	s = pstrdup("hello world");
	pfree(s);

	MemoryContextSwitchTo(old_context);

	PG_RETURN_VOID();
}

Datum
hoge_list(PG_FUNCTION_ARGS)
{
	int i = PG_GETARG_INT32(0);
	ListCell *lc;
	MemoryContext old_context;

	old_context = MemoryContextSwitchTo(TopTransactionContext);
	my_local_dsa_context = CreateDsaLocalContext(my_shared_dsa_context);
	MemoryContextSwitchTo(old_context);

	old_context = MemoryContextSwitchTo(my_local_dsa_context);

	/* Manipulate a list in shared memory. */
	LWLockAcquire(AddinShmemInitLock, LW_EXCLUSIVE);
	if (i < 0)
		*my_list = list_delete_int(*my_list, -i);
	else
		*my_list = lappend_int(*my_list, i);
	LWLockRelease(AddinShmemInitLock);

	SetDsaSharedContext(my_local_dsa_context, my_shared_dsa_context);

	/* Dump list. */
	elog(NOTICE, "Contents of list:");
	foreach(lc, *my_list)
		elog(NOTICE, " %d", lfirst_int(lc));

	MemoryContextSwitchTo(old_context);

	PG_RETURN_VOID();
}


/*
 * Error test to check chunk is dsa_freed
 * Currently after this function is called by user
 * process goes down because it tries to access dangling pointer
 */
Datum
hoge_list_error(PG_FUNCTION_ARGS)
{
	int i = PG_GETARG_INT32(0);
	ListCell *lc;
	MemoryContext old_context;

	old_context = MemoryContextSwitchTo(TopTransactionContext);
	my_local_dsa_context = CreateDsaLocalContext(my_shared_dsa_context);
	MemoryContextSwitchTo(old_context);

	old_context = MemoryContextSwitchTo(my_local_dsa_context);

	/* Manipulate a list in shared memory. */
	LWLockAcquire(AddinShmemInitLock, LW_EXCLUSIVE);
	if (i < 0)
		*my_list = list_delete_int(*my_list, -i);
	else
		*my_list = lappend_int(*my_list, i);

	elog(ERROR, "error test");
	LWLockRelease(AddinShmemInitLock);
	SetDsaSharedContext(my_local_dsa_context, my_shared_dsa_context);

	/* Dump list. */
	elog(NOTICE, "Contents of list:");
	foreach(lc, *my_list)
		elog(NOTICE, " %d", lfirst_int(lc));

	MemoryContextSwitchTo(old_context);

	PG_RETURN_VOID();
}


/* Support code to make a dsa_area into a MemoryContext. */
typedef struct dsa_temp_block dsa_temp_block;

/* this block is used to free all "local" dsa chunks */
struct dsa_temp_block
{
	dsa_temp_block *next; /* single linked list */
	int idx; 			/* index of array of pointer */
	dsa_pointer chunks[NUM_CHUNKS]; /* relative address of chunks */
};

struct hoge_memory_context
{
	struct MemoryContextData memory_context;
	void	 *base;
	dsa_area   *area;
	/* array of pointer to dsa_allocated chunks to be freed if error occurs */
	dsa_temp_block *temp_block;  
};


static void *
hoge_alloc(MemoryContext context, Size size)
{
	struct hoge_memory_context *c = (struct hoge_memory_context *) context;	
	char *chunk_backp;	
	dsa_temp_block *new_block;

	/* we only allow palloc in temporary DSA-MemoryContext */
	if (!c->temp_block)
	{
		elog(ERROR, "hoge_alloc should be run in "
			 "temporary DSA MemoryContext");
		return NULL;
	}

	if (++c->temp_block->idx == NUM_CHUNKS)
	{
		MemoryContext old_context = MemoryContextSwitchTo(TopTransactionContext);
		
		/* New block is inserted at head position */
		new_block = (dsa_temp_block *) palloc0(sizeof(dsa_temp_block));
		new_block->next = c->temp_block;
		c->temp_block = new_block->next;
		
		MemoryContextSwitchTo(old_context);
	}

	/* Add space for the secret context pointer. */
	c->temp_block->chunks[c->temp_block->idx] = dsa_allocate(c->area, sizeof(void *) + size);
	chunk_backp = (char *)c->base + c->temp_block->chunks[c->temp_block->idx];
	*(void **) chunk_backp = context;

	return chunk_backp + sizeof(void *);
}

static void
hoge_free(MemoryContext context, void *pointer)
{
	struct hoge_memory_context *c = (struct hoge_memory_context *) context;
	dsa_temp_block *cur;
	char *chunk_backp;
	dsa_pointer dp;

	/* Rewind to the secret start of the chunk */
	chunk_backp = (char *) pointer - sizeof(void *);
	dp = (dsa_pointer) ((char *)chunk_backp - (char *)c->base);

	dsa_free(c->area, dp);

	/* ealy return if Context is permanent */
	if (!c->temp_block)
		return;
		
	/* To avoid free twice at MemoryContextDelete we remove it from */
	for (cur = c->temp_block; cur != NULL; cur = cur->next)
	{
		for (cur->idx = 0; cur->idx < NUM_CHUNKS; cur->idx++) 
		{
			if (cur->chunks[cur->idx] == dp)
			{	
				cur->chunks[cur->idx] = 0;
				break;
			}
		}
	}
}

static void *
hoge_realloc(MemoryContext context, void *pointer, Size size)
{
	elog(ERROR, "hoge_realloc not implemented");
	return NULL;
}

static void
hoge_reset(MemoryContext context)
{
	elog(ERROR, "hoge_reset not implemented");
}

static void
hoge_delete(MemoryContext context)
{	
	struct hoge_memory_context *c = (struct hoge_memory_context *) context;
	dsa_temp_block *cur = c->temp_block;

	/* We don't support MemoryContextDelete if Context is permanent */
	if (!c->temp_block)
	   elog(ERROR, "hoge_delete is not supported at permanent DSA MemoryContext");

	/* free all chunks and blocks */
	while (cur && cur->chunks)
   	{
		dsa_temp_block *next = cur->next;
		
		for (cur->idx = 0; cur->idx < NUM_CHUNKS; ++cur->idx) 
		{
			/* Rewind to the secret start of the chunk */
			if (cur->chunks[cur->idx] != 0)
			{
				dsa_free(c->area, cur->chunks[cur->idx]);
				elog(NOTICE, "chunk is dsa_freed");
			}
		}
		pfree(cur);
		cur = next;
	}
	/* Finally, free the context header */
	pfree(c);
}

static Size
hoge_get_chunk_space(MemoryContext context, void *pointer)
{
	elog(ERROR, "hoge_get_chunk_space not implemented");
	return 0;
}

static bool
hoge_is_empty(MemoryContext context)
{
	elog(ERROR, "hoge_is_empty not implemented");
	return false;
}

static void
hoge_set_state(MemoryContext context,
			   MemoryStatsPrintFunc printfunc, void *passthru,
			   MemoryContextCounters *totals)
{
	elog(ERROR, "hoge_set_state not implemented");
}

static void
hoge_set_check(MemoryContext context)
{
}

MemoryContext
make_hoge_memory_context(dsa_area *area, void *base, bool isLocal)
{
	static const MemoryContextMethods hoge_methods = {
		hoge_alloc,
		hoge_free,
		hoge_realloc,
		hoge_reset,
		hoge_delete,
		hoge_get_chunk_space,
		hoge_is_empty,
		hoge_set_state,
#ifdef MEMORY_CONTEXT_CHECKING
		hoge_set_check
#endif
	};

	struct hoge_memory_context *result =
		palloc0(sizeof(struct hoge_memory_context));

	MemoryContextCreate(&result->memory_context,
						T_SlabContext, /* TODO: this is a lie */
						&hoge_methods,
						NULL,
						"hoge");
	result->base = base;
	result->area = area;
	
	/* If context is shared one, chunks become permanent so temp_block is NULL */
	if (isLocal)
		result->temp_block = (dsa_temp_block *) palloc0(sizeof(dsa_temp_block));
	else
		result->temp_block = NULL;

	return &result->memory_context;
}

MemoryContext
CreateDsaLocalContext(MemoryContext shared)
{
	MemoryContext local;
	struct hoge_memory_context *shared_context = (struct hoge_memory_context *) shared;

	AssertArg(MemoryContextIsValid(shared));
	
	local = make_hoge_memory_context(shared_context->area, shared_context->base, true);
	MemoryContextSetParent(local, TopTransactionContext);
	
	Assert(MemoryContextIsValid(local));

	return local;
}

/* 
 * SetDsaSharedContext
 *
 * We don't want to leak memory in shared memory. Unlike local process, 
 * memory leak still exists even after local process is terminated.
 * If error occurs in transaction, we free all dsa_allocated chunks linked
 * from local DSA-MemoryContext. When the function using DSA-MemoryContext make
 * sure that the memory leak does not happen, SetDsaSharedContext should be called.
 *
 * XXX: Assuming following case: when catcache is inserted into hash table on 
 * shared memory and transaction is aborted we wouldn't leak memory because 
 * some eviction alogorithm would clean up useless cache eventually.
 *
 */
void
SetDsaSharedContext(MemoryContext local, MemoryContext shared)
{
	MemoryContext old_context;

	struct hoge_memory_context *local_dsa_cxt = (struct hoge_memory_context *) local;
	dsa_temp_block *cur = local_dsa_cxt->temp_block;

	AssertArg(MemoryContextIsValid(local));
	AssertArg(MemoryContextIsValid(shared));
	

	/* change backpointer to shared MemoryContext */	
	while (cur && cur->chunks)
   	{
		dsa_temp_block *next = cur->next;
		
		for (cur->idx = 0; cur->idx < NUM_CHUNKS; ++cur->idx) 
		{
			/* Rewind to the secret start of the chunk */
			if (cur->chunks[cur->idx] != 0)
				*(void **)( cur->chunks[cur->idx] + (char *)local_dsa_cxt->base)
						   = shared;
		}
		pfree(cur);
		cur = next;
	}

	old_context = MemoryContextSwitchTo(TopTransactionContext);

	/* rebuild temporary block list again */
	local_dsa_cxt->temp_block = (dsa_temp_block *) palloc0(sizeof(dsa_temp_block));

	MemoryContextSwitchTo(old_context);
}
