/*
 * AUTHOR:
 *   Bernd Helmle
 * DESCRIPTION:
 *   Functions to translate a SQL92 compliant view definition
 *   into update rules.
 * TODO:
 *   - conditional rule (WITH CHECK OPTION)
 *   - CHECK CASCADE / LOCAL OPTION
 *   - checkTree() improvements
 *   - Make code to work on views based on views ...
 *   - View declarations like SELECT a AS b from foo doesn't work
 */

#ifndef VIEW_UPDATE_H
#define VIEW_UPDATE_H

#include <postgres.h>
#include <fmgr.h>
#include <catalog/pg_type.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_rewrite.h>
#include <executor/spi.h>
#include <lib/stringinfo.h>
#include <unistd.h>
#include <nodes/makefuncs.h>
#include <nodes/print.h>
#include <rewrite/rewriteSupport.h>
#include <rewrite/rewriteDefine.h>
#include <parser/parsetree.h>
#include <utils/lsyscache.h>
#include <utils/syscache.h>
#include <utils/errcodes.h>
#include <parser/parse_func.h>
#include <parser/parse_oper.h>

#define __DEBUG_ON

#ifdef __DEBUG_ON
#define VERBOSE_MESSAGE(a) elog( DEBUG1, (a) )
#define WAIT(a) wait_for_debugger( (a) )
#else
#define VERBOSE_MESSAGE(a) //dummy
#define WAIT(a)
#endif

/*------------------------------------------------------------------------------
 * some names to be used for implicit rules
 *------------------------------------------------------------------------------
 */
#define INSERTRULENAME "_INSERT"
#define DELETERULENAME "_DELETE"
#define UPDATERULENAME "_UPDATE"
#define NOTHING_INSERTRULENAME "_NOTHING_INSERT"
#define NOTHING_UPDATERULENAME "_NOTHING_UPDATE"
#define NOTHING_DELETERULENAME "_NOTHING_DELETE"

struct ViewBaseRelationItem {

	Oid           reloid;  /* OID for this base relation */
	List          *childs; /* child relations, NIL if no children */
	Query         *rule;   /* _RETURN rule of a view relation */
	TargetEntry   **tentries; /* saves order of column target list */
	Form_pg_class rel;     /* pg_class structure of relation */
	bool          isView;  /* relation is a view */

};

struct ViewBaseRelation {

	Oid fatherRelation;  /* Oid of father relation, 0 indicates root */
	List *defs;          /* relation and, if relation is a view, return rule */

};

/*------------------------------------------------------------------------------
 * DEBUG function
 *------------------------------------------------------------------------------
 */

static int debug_wait = 40;
static __inline__ void 
wait_for_debugger(int32 val) {

	if (val > 0)
		sleep( val );
  
}

extern void
read_rearranged_colls( struct ViewBaseRelation *tree );

extern bool
check_reltree( struct ViewBaseRelation *node,
			   List   **baserelations );

extern bool
checkTree( const Query *query,
		   struct ViewBaseRelation *tree );

extern Oid 
get_reloid_from_select( const Query         *select,
						RangeTblEntry **rel_entry);

extern Query *
transform_select_to_update( const Query    *update,
							const Relation rel,
							const RangeVar       *var,
							TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade );

extern Query *
transform_select_to_insert( const Query    *select,
							const Relation rel,
							const RangeVar       *var,
							TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade );

extern Query *
transform_select_to_delete( const Query    *delete,
							const Relation rel,
							const RangeVar       *var,
							TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade );

void
get_base_base_relations( const Query *view,
						 Oid         baserel,
						 List        **list );

extern void
copyReversedTargetEntryPtr( List        *targetList,
							TargetEntry *targets[] );

#endif // VIEW_UPDATE_H
