/*
 * Generic array aggregator
 *
 * Originally based on code by:
 * Mark L. Woodward
 * DMN Digital Music Network.
 * www.dmn.com
 *
 * $PostgreSQL: pgsql/contrib/intagg/int_aggregate.c,v 1.27 2008/05/12 00:00:42 alvherre Exp $
 *
 * Copyright (C) Digital Music Network
 * December 20, 2001
 *
 * This file is the property of the Digital Music Network (DMN).
 * It is being made available to users of the PostgreSQL system
 * under the BSD license.
 */
#include "postgres.h"

#include "executor/executor.h"
#include "fmgr.h"
#include "utils/array.h"
#include "utils/memutils.h"
#include "utils/lsyscache.h"

PG_MODULE_MAGIC;

#define START_NUM	(MAXALIGN(32)*8)	/* initial size of null bitmap, in bits */
#define START_SIZE	1024				/* initial size of memory block */

Datum		array_agg_trans(PG_FUNCTION_ARGS);
Datum		array_agg_final(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(array_agg_trans);
PG_FUNCTION_INFO_V1(array_agg_final);

typedef struct ArrayAggState
{
	bool	 typbyval;
	char	 typalign;
	int16	 typlen;
	uint32	 endoffset; 
} ArrayAggState;

/*
 * Manage the allocation state of the array
 *
 * Note that the array needs to be in a reasonably long-lived context,
 * ie the Agg node's aggcontext.
 */

/* Called for each iteration during an aggregate function */
Datum
array_agg_trans(PG_FUNCTION_ARGS)
{
	AggState		*aggstate = (AggState *) fcinfo->context;
	Oid				 element_type = get_fn_expr_argtype(fcinfo->flinfo, 1);
	ArrayType		*p;
	ArrayAggState	*state;
	bits8			*nullptr;
	Datum			 element;
	int32			 statepos;
	int32			 startoffset;
	int32			 new_endoffset;				
	int32			 extra_null_len;

	/*
	 * As of PG 8.1 we can actually verify that we are being used as an
	 * aggregate function, and so it is safe to scribble on our left input.
	 */
	if (!(aggstate && IsA(aggstate, AggState)))
		elog(ERROR, "array_agg_trans may only be used as an aggregate");

	if (PG_ARGISNULL(0))
	{
		/* First time through */
		p = (ArrayType *) MemoryContextAlloc(aggstate->aggcontext, START_SIZE);
		SET_VARSIZE(p, START_SIZE);
		p->ndim = 1;
		p->dataoffset = ARR_OVERHEAD_WITHNULLS(1, START_NUM)
						+ MAXALIGN(sizeof(ArrayAggState));
		p->elemtype = element_type;
		ARR_DIMS(p)[0] = 0;
		ARR_LBOUND(p)[0] = 1;
		nullptr = ARR_NULLBITMAP(p);
		memset(nullptr, 0, START_NUM / 8);
		state = (ArrayAggState *) (ARR_DATA_PTR(p) - MAXALIGN(sizeof(ArrayAggState)));
		get_typlenbyvalalign(element_type, &state->typlen, &state->typbyval, &state->typalign);
		state->endoffset = p->dataoffset;
	}
	else
	{
		p = PG_GETARG_ARRAYTYPE_P(0);

		if (p->ndim != 1)
			elog(ERROR, "Input array must be one-dimensional");

		else if (p->dataoffset == 0 || p->dataoffset == ARR_OVERHEAD_WITHNULLS(1, ARR_DIMS(p)[0]))
		{
			/* This array doesn't have any extra space between the null bitmap
			 * (if any) and the actual data, so assume it is an initial aggregate
			 * value */

			elog(ERROR, "An initial transition state must not be specified to array_agg_trans");
		}

		state = (ArrayAggState *) (ARR_DATA_PTR(p) - MAXALIGN(sizeof(ArrayAggState)));
		nullptr = ARR_NULLBITMAP(p) + (ARR_DIMS(p)[0] / 8);
	}

	startoffset = state->endoffset;

	if (PG_ARGISNULL(1))
	{
		element = (Datum) 0; /* Silence compiler warning */
		new_endoffset = startoffset;
	}
	else
	{
		element = PG_GETARG_DATUM(1);

		if (state->typlen < 0)
			element = PointerGetDatum(PG_DETOAST_DATUM(element));

		startoffset = att_align_datum(startoffset, state->typalign, state->typlen, element);
		new_endoffset = att_addlength_datum(startoffset, state->typlen, element);
	}

	if (nullptr >= (bits8 *) state)
	{
		/* The null pointer array is full - will need to double its size */
		extra_null_len = ARR_DIMS(p)[0] / 8;
	}
	else
		extra_null_len = 0;


	/* Ensure array has space for another item */
	if (new_endoffset + extra_null_len > ARR_SIZE(p))
	{
		ArrayType  *pn;
		int			n = ARR_SIZE(p);

		do
		{
			n *= 2;
			/* check for overflow of total request */
			if (!AllocSizeIsValid(n))
				ereport(ERROR,
						(errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
						 errmsg("array size exceeds the maximum allowed (%d)",
								(int) MaxAllocSize)));
		}
		while (new_endoffset + extra_null_len > n);


		pn = (ArrayType *) MemoryContextAlloc(aggstate->aggcontext, n);
		statepos = (char *) state - (char *) p;
		if (extra_null_len)
		{
			memcpy(pn, p, statepos);
			memset((char *) pn + statepos, 0, extra_null_len);
			memcpy((char *) pn + statepos + extra_null_len, state, VARSIZE(p) - statepos);
			pn->dataoffset += extra_null_len;
		}
		else
		{
			memcpy(pn, p, VARSIZE(p));
		}
		SET_VARSIZE(pn, n);
		/* do not pfree(p), because nodeAgg.c will */
		p = pn;
		state = (ArrayAggState *) ((char *) p + statepos + extra_null_len);
		nullptr = ARR_NULLBITMAP(p) + (ARR_DIMS(p)[0] / 8);
	} 
	else if (extra_null_len)
	{
		/* The null bitmap is full, but there is enough memory allocated to extend it */
		statepos = (char *) state - (char *) p;
		/* Relocate the data */
		memmove((char *) state + extra_null_len, state, state->endoffset - statepos);
		/* Clear the new null array */
		memset(state, 0, extra_null_len);

		state = (ArrayAggState *) ((char *) p + statepos + extra_null_len);
		p->dataoffset += extra_null_len;
	}

	startoffset += extra_null_len;
	new_endoffset += extra_null_len;

	if (!PG_ARGISNULL(1))
	{
		if (state->typbyval)
			store_att_byval((char *) p + startoffset, element, state->typlen);
		else
			memcpy((char *) p + startoffset, DatumGetPointer(element), new_endoffset - startoffset);
		*nullptr |= 1 << (ARR_DIMS(p)[0] % 8);
	}

	ARR_DIMS(p)[0]++;
	state->endoffset = new_endoffset;

	PG_RETURN_ARRAYTYPE_P(p);
}

/*
 * This is the final function used for the integer aggregator. It returns all
 * the integers collected as a one dimensional integer array
 */
Datum
array_agg_final(PG_FUNCTION_ARGS)
{
	ArrayType	*p;
	ArrayType	*pnew;
	int16		 typlen;
	bool		 typbyval;
	char		 typalign;
	int			 overhead;
	int			 datalen;
	bits8		*nullbitmap;
	int			 bitmask;
	int			 i;
	bool		 hasnulls;
	char		*ptr;

	/*
	 * As of PG 8.1 we can actually verify that we are being used as an
	 * aggregate function, and so it is safe to scribble on our left input.
	 */
	if (!(fcinfo->context && IsA(fcinfo->context, AggState)))
		elog(ERROR, "array_agg_final may only be used as an aggregate");

	p = PG_GETARG_ARRAYTYPE_P(0);

	if (ARR_NDIM(p) != 1)
		elog(ERROR, "Array input to array_agg_final must have one dimension");
	
	get_typlenbyvalalign(ARR_ELEMTYPE(p), &typlen, &typbyval, &typalign);

	ptr = ARR_DATA_PTR(p);
	nullbitmap = ARR_NULLBITMAP(p);
	bitmask = 1;
	hasnulls = 0;

	for (i = 0; i < ARR_DIMS(p)[0]; i++)
	{
		if (*nullbitmap & bitmask)
		{
			ptr = att_addlength_pointer(ptr, typlen, ptr);
			ptr = (char *) att_align_nominal(ptr, typalign);
		}
		else
			hasnulls = 1;
		bitmask <<= 1;
		if (bitmask == 0x100)
		{
			nullbitmap++;
			bitmask = 1;
		}
	}
	/* get target size */
	if (hasnulls)
		overhead = ARR_OVERHEAD_WITHNULLS(1, ARR_DIMS(p)[0]);
	else
		overhead = ARR_OVERHEAD_NONULLS(1);
	datalen = ptr - ARR_DATA_PTR(p);

	/* use current transaction context */
	pnew = palloc(overhead + datalen);

	/* Copy the header over */
	memcpy(pnew, p, overhead);

	/* fix up the fields in the new array to match normal conventions */
	SET_VARSIZE(pnew, overhead + datalen);
	pnew->dataoffset = hasnulls ? overhead : 0;

	/* Copy the data over */
	memcpy(ARR_DATA_PTR(pnew), ARR_DATA_PTR(p), datalen);

	/* do not pfree(p), because nodeAgg.c will */

	PG_RETURN_ARRAYTYPE_P(pnew);
}

