/*
** ident-des.c
**
**      This file unencrypts encrypted IDENT responses.
**      It is largely based on code taken from the pidentd source,
**      the copyright for which is found below.
**
**      Portions Copyright (c) 2002 David M. Kaplan <dmkaplan@ucdavis.edu> GPL
**
**
** idecrypt.c - Encrypted IDENT response decryption utility.
**
** Copyright (c) 1997-2000 Peter Eriksson <pen@lysator.liu.se>
**
** This program is free software; you can redistribute it and/or
** modify it as you wish - as long as you don't claim that you wrote
** it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

/**************************************************
 * These should be added to some configure routine.
 **************************************************/
#define HAVE_LIBDES 1
/* #ifdef HAVE_DES_H */
#define HAVE_OPENSSL_DES_H 1
#define HAVE_LIBCRYPTO 1
/*************************************************/

#ifdef HAVE_LIBDES

#include <string.h>
#include <time.h>
#include <pwd.h>

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sys/stat.h>
#include <sys/time.h>

#ifdef HAVE_DES_H
#include <des.h>
#elif HAVE_OPENSSL_DES_H
#include <openssl/des.h>
#endif

#include "postgres.h"
#include "miscadmin.h"
#include "libpq/hba.h"

/* Name of file with identd DES keys */
#define IDENT_KEY_FILE "pg_identd.key"

/* Time before packet expires - currently 1 hour */
#define IDENT_DES_TIME 3600


/* pdes.h */
/*
** pdes.h - Pidentd DES encryption stuff
**
** Copyright (c) 1997-1999 Peter Eriksson <pen@lysator.liu.se>
**
** This program is free software; you can redistribute it and/or
** modify it as you wish - as long as you don't claim that you wrote
** it.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

struct des_info
{
    uint32_t checksum;
    uint16_t random;
    /* FIXME: uid_t isn't necessarily short.  */
    uint16_t uid;
    uint32_t date;
    uint32_t ip_local;
    uint32_t ip_remote;
    uint16_t port_local;
    uint16_t port_remote;
};

typedef union des_data
{
    struct des_info   fields;
    uint32_t        longs[6];
    unsigned char chars[24];
} data;


/* idecrypt.c */
static char
is_base_64 [] =
{
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
    0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
    0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

static unsigned char
to_bin[] =
{
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x3e, 0x80, 0x80, 0x80, 0x3f,
    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
    0x3c, 0x3d, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
    0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
    0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
    0x17, 0x18, 0x19, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
    0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
    0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
    0x31, 0x32, 0x33, 0x80, 0x80, 0x80, 0x80, 0x80,
};


/*
 * identd_key_getfilename --- get full pathname of identd.key file
 *
 * Note that result string is palloc'd, and should be freed by the caller.
 */
char *
ident_key_getfilename(void)
{
	int			bufsize;
	char	   *pfnam;

	bufsize = strlen(DataDir) + strlen(IDENT_KEY_FILE) + 2;
	pfnam = (char *) palloc(bufsize);
	snprintf(pfnam, bufsize, "%s/%s", DataDir, IDENT_KEY_FILE);

	return pfnam;
}



bool 
ident_decrypt_packet(unsigned char *packet, union des_data *r)
{
    int i, j;
    char keybuf[1024+1];
    int keyfile_fd;
    des_cblock key_bin;
    des_key_schedule sched;
    
    char *keyfile_path = ident_key_getfilename(); /* get key full path. */

    keyfile_fd = open(keyfile_path, O_RDONLY);
    if (keyfile_fd < 0)
      elog(FATAL,"IDENT key file not found at: %s",keyfile_path);

    pfree(keyfile_path); /* Free filename mem */


    /* Try to decrypt with each key found in the key file */
    while (read(keyfile_fd, keybuf, sizeof(keybuf)-1) == sizeof(keybuf)-1)
    {
	keybuf[sizeof(keybuf)-1] = '\0';
	des_string_to_key(keybuf, &key_bin);
	des_set_key(&key_bin, sched);
	
	
	for (i = 0, j = 0; i < 24; i += 3, j += 4)
	{
	    r->chars[i  ] = (to_bin[packet[j  ]] << 2) + (to_bin[packet[j+1]] >> 4);
	    r->chars[i+1] = (to_bin[packet[j+1]] << 4) + (to_bin[packet[j+2]] >> 2);
	    r->chars[i+2] = (to_bin[packet[j+2]] << 6) + (to_bin[packet[j+3]]);
	}
    
	des_ecb_encrypt((des_cblock *)&(r->longs[4]),
			(des_cblock *)&(r->longs[4]),
			sched, DES_DECRYPT);
	r->longs[4] ^= r->longs[2];
	r->longs[5] ^= r->longs[3];
	
	des_ecb_encrypt((des_cblock *)&(r->longs[2]),
			(des_cblock *)&(r->longs[2]),
			sched, DES_DECRYPT);
	
	r->longs[2] ^= r->longs[0];
	r->longs[3] ^= r->longs[1]; 
	des_ecb_encrypt((des_cblock *)&(r->longs[0]),
			(des_cblock *)&(r->longs[0]),
			sched, DES_DECRYPT);

	for (i = 1; i < 6; i++)
	{
	    r->longs[0] ^= r->longs[i];
	}
	
	if (r->fields.checksum == 0)
	    goto GoodKey;
    }
    close(keyfile_fd);
    return false;
    
  GoodKey:
    close(keyfile_fd);
    return true;
}

bool
ident_des(const struct in_addr remote_ip_addr,
	 const struct in_addr local_ip_addr,
	 const ushort remote_port,
	 const ushort local_port,
	 char *ident_user)
{
  int i, c;
  union des_data r;
  unsigned char packet[32];
  struct passwd *pw;

  /* ident_user must be big enough for packet and brackets. */
  if (strlen(ident_user) < 34) return false; 

  /* Must begin and end with brackets. */
  if ( (ident_user[0] != '[') || (ident_user[33] != ']') ) return false;

  /* Copy over packet. */
  for (i = 0; i < 32; i++)
    {
      c = ident_user[i+1];
      if (c < 0 || c > 255)
	break;
      if (!is_base_64[c])
	break;
      packet[i] = c;
    }
  
  /* Packet had bad characters. */
  if (i < 32) return false;

  /* Decrypt packet.  If bad packet, abandon. */
  if (!ident_decrypt_packet( packet, &r )) return false;
  
  /* Check values returned. Switched as IDENTD runs on client, not server. */
  if ( (r.fields.ip_local != remote_ip_addr.s_addr) ||
       (r.fields.ip_remote != local_ip_addr.s_addr) ||
       (r.fields.port_local != remote_port)  ||
       (r.fields.port_remote != local_port)  ||
       (ntohl(r.fields.date) - time(NULL) > IDENT_DES_TIME) )
    return false;
  
  /* Obtain username (on server) from UID (on client). */
  pw = getpwuid(ntohs(r.fields.uid));
  
  /* Error if UID doesnt match anyone on pg server machine. */
  if (pw == NULL)
    {
      elog(LOG, "ident_des: unknown user on postgresql server with uid %d",
	   (int) ntohs(r.fields.uid));
      return false;
    }

  /* Copy username over. */
  StrNCpy(ident_user, pw->pw_name, IDENT_USERNAME_MAX + 1);

  return true;
}

#else /* no HAVE_LIBDES */

#error Need a DES library to compile Idecrupt.

#endif
